/*
 *************************************************************************
 * Copyright (C) 2017, Cisco Systems
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 *************************************************************************
 *
 *  File:    defines.h
 *  Author:  Koushik Chakravarty <kouchakr@cisco.com>
 *
 ************************************************************************
 *
 *  This file contains part of the module wide definitions
 *
 ************************************************************************
 */
#ifndef __DEFINES_H__
#define __DEFINES_H__

#ifdef NVM_BPF_KERNEL
#include "vmlinux.h"
#else
#ifdef NVM_BPF_USERSPACE
#include <netinet/tcp.h>
#include <netinet/udp.h>
#include <netinet/ip.h>
#include <netinet/ip6.h>
#include <arpa/inet.h>
#include <linux/types.h>
#include <asm/byteorder.h>
#include <stdlib.h>
#else
#include <linux/tcp.h>
#include <linux/udp.h>
#include <linux/ip.h>
#include <linux/ipv6.h>
#endif
#endif

#ifndef ARRAY_SIZE
#define ARRAY_SIZE(a) \
	(sizeof(a)/sizeof(a[0]))
#endif

/*
 * per RFC 1034, the total number of octets that represent a domain name
 * (i.e., the sum of all label octets and label lengths) is limited to 255
 */
#define MAX_DOMAIN_NAME_SIZE    255

/* Memory allocation macros */
#ifdef NVM_BPF_USERSPACE
#define KMALLOC(nBytes) malloc(nBytes)
#define KMALLOC_ATOMIC(nBytes) malloc(nBytes)
#define KFREE(pBlock) \
	do { if (pBlock) { free(pBlock); pBlock = NULL; } } while(0)
#else
#define KMALLOC(nBytes) kmalloc(nBytes, GFP_KERNEL)
#define KMALLOC_ATOMIC(nBytes) kmalloc(nBytes, GFP_ATOMIC)
#define KFREE(pBlock) \
	{ \
		kfree(pBlock); \
		pBlock = NULL; \
	}
#endif

typedef enum _error_code {
#ifdef NVM_BPF_KERNEL
	ERROR_SUCCESS = 0,
#else
	SUCCESS = 0,
#endif
	ERROR_ERROR = 1,
	ERROR_ALREADY_INITIALIZED = 2,
	ERROR_BAD_PARAM = 3,
	ERROR_INSUFFICIENT_RESOURCES = 4,
	ERROR_UNEXPECTED = 5,
	ERROR_NOT_INITIALIZED = 6,
	ERROR_NOT_SUPPORTED = 7,
} error_code;

enum userspace_map_key
{
    VERBOSE_KEY = 0,
    VPN_INTERFACE_KEY
};

#ifndef AF_INET
#define AF_INET 2
#endif

#ifndef AF_INET6
#define AF_INET6 10
#endif

#ifndef IPPROTO_TCP
#define IPPROTO_TCP 6
#endif

#ifndef IPPROTO_UDP
#define IPPROTO_UDP 17
#endif

#ifndef SOCK_DGRAM
#define	SOCK_DGRAM 2
#endif

enum direction_e {
#ifndef NVM_BPF_KERNEL // it is already defined in vmlinux.h
	UNKNOWN = -1,
#endif
	INBOUND = 0,
	OUTBOUND = 1
};

#ifdef NVM_BPF_USERSPACE
// Custom IPv6 header structure that matches kernel's struct ipv6hdr
struct ipv6hdr {
#if defined(__LITTLE_ENDIAN_BITFIELD)
	__u8			priority:4,
				version:4;
#elif defined(__BIG_ENDIAN_BITFIELD)
	__u8			version:4,
				priority:4;
#else
#error	"Please fix <asm/byteorder.h>"
#endif
    __u8            flow_lbl[3];
    __be16          payload_len;
    __u8            nexthdr;
    __u8            hop_limit;
    struct in6_addr saddr;
    struct in6_addr daddr;
};
#endif

struct ip_hdr {
	union {
		struct iphdr h_ip4;
		struct ipv6hdr h_ip6;
	};
};

struct l4_hdr {
	union {
		struct udphdr h_udp;
		struct tcphdr h_tcp;
	};
};

struct nw_pkt_meta {
#ifdef NVM_BPF_USERSPACE
	uint64_t pid;
	uint64_t process_creation_time;
#endif
	struct ip_hdr *ip_header;
	struct l4_hdr *l4_header;
	enum direction_e direction;
};

struct bpf_message_header {
	uint16_t length;
	uint8_t version;
};

struct bpf_nw_pkt_meta {
	struct bpf_message_header header;
	uint64_t pid;
	uint64_t process_creation_time;
	enum direction_e direction;
	struct ip_hdr ip_header;
	struct l4_hdr l4_header;
};

#define HTON32(v) \
	(v>>24 | (v>>16 & 0xff) << 8 | (v>>8 & 0xff) << 16 | (v & 0xff) << 24)
#define NTOH32(v) HTON32(v)
#define HTON16(v) ((v >> 8) | ((v & 0xff) << 8))
#define NTOH16(v) HTON16(v)

#endif
